## Advanced Events


We have already seen how to use simplified event handling in the previous section. In this section, we will learn how to use advanced event handling.

### Standard Event Handling

Event handling in devflow can use a more complex handler that takes exactly one argument ```Event```, it contains event type, event data, event keys and indexes. keys and indexes are used in template components, which will be covered later.

Event handlers supported by a component are stored in component attributes with ```event_``` prefix. ```event_``` prefix is followed by event type, e.g. ```event_click``` is the click event of a button.

* ```on```: register a callback function to an event. The callback function is simplified.

* ```on_standard```: register a callback function to an event. The callback function takes exactly one argument ```Event```.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        btn = mui.Button("A Button")
        btn2 = mui.Button("A Button 2")

        btn.event_click.on(self._on_button_clicked)
        btn2.event_click.on_standard(self._on_button_click_standard_event)

        return mui.VBox([
            btn,
            btn2,
        ])

    async def _on_button_clicked(self):
        print("Button Clicked")

    async def _on_button_click_standard_event(self, ev: mui.Event):
        print(ev)

```

There are two special components that support all pointer events: ```mui.FlexBox``` and ```mui.Image```. 

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        box_with_ev = mui.HBox([
            mui.Button("Button")
        ]).prop(width="100px", height="100px", backgroundColor="red", alignItems="center")
        box_with_ev.event_pointer_move.on(self._on_pointer_move)
        self.event_content = mui.JsonViewer()
        return mui.VBox([
            box_with_ev,
            self.event_content,
        ])

    async def _on_pointer_move(self, data):
        await self.event_content.send_and_wait(self.event_content.update_event(data=data))

```

We can see that there are much events sent to backend when handle event ```PointerMove```. we can reduce the number of events by using ```debounce``` or ```throttle```.

* debounce: the event handler will be called after the event stops for a specified time.

* throttle: the event handler will be called at most once in a specified time.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        box_with_ev = mui.HBox([
        ]).prop(width="100px", height="100px", backgroundColor="red")
        box_with_ev2 = mui.HBox([
        ]).prop(width="100px", height="100px", backgroundColor="blue")

        box_with_ev.event_pointer_move.on(self._on_pointer_move).configure(debounce=500) # ms
        box_with_ev2.event_pointer_move.on(self._on_pointer_move2).configure(throttle=500) # ms

        self.event_content = mui.JsonViewer()
        return mui.VBox([
            mui.HBox([
                box_with_ev,
                box_with_ev2,
            ]),
            self.event_content,
        ])

    async def _on_pointer_move(self, data):
        await self.event_content.send_and_wait(self.event_content.update_event(data=data))
    async def _on_pointer_move2(self, data):
        print(2)
        await self.event_content.send_and_wait(self.event_content.update_event(data=data))

```